//=============================================================================
// SlipDamageBalancer.js
// ----------------------------------------------------------------------------
// (C)2022 kuroame koubou
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 0.9.0 2023/01/11 初版
// ----------------------------------------------------------------------------
// [Blog]   : http://kuroamekoubou.blog.fc2.com
// [Twitter]: https://twitter.com/kuroVelter9623
//=============================================================================

/*:
 * @plugindesc スリップダメージバランサープラグイン
 * @target MZ
 * @author くろあめ
 *
 * @param SlipDamAnime
 * @text スリップダメージ時アニメ
 * @desc 負のスリップダメージを受けるときに再生するアニメのODを指定します。0指定で再生を無効にします。
 * @default 0
 * @type number
 *
 * @help
 * このスクリプトは、HP再生やスリップダメージを実行する
 * 「regenerateHp」関数を再定義します
 * 都合上、極力上の方に配置することを推奨します。
 * 
 * 指定のメタタグをメモ欄に記載することで
 * その要素を持つアクター/エネミーが、スリップダメージ（負のHP再生）を受ける際に
 * その値を倍加・軽減します。
 * 
 * 倍化・軽減が行われるのは、スリップダメージのみです
 * HP再生に対しては適用されません。
 * 
 * また、スリップダメージ発生時に、アニメを再生することができます。
 *
 *
 * アクター、エネミー、職業、装備、ステートのメモ欄に
 * <SlipDivision:x>と記載することで 
 * その要素を持つアクター/エネミーが、スリップダメージを受ける際に
 * 最終ダメージをXで割ります。
 *
 * アクター、エネミー、職業、装備、ステートのメモ欄に
 * <SlipPush:x>と記載することで 
 * その要素を持つアクター/エネミーが、スリップダメージを受ける際に
 * 最終ダメージをX倍にします。
 *
 * 利用形態（商用、18禁利用等）について制限はありません
 * 作者への使用報告や、ReadMeなどへ作者・サイト名の記載も任意となります。
 *
 * ただし、作者はこれらのプラグインについて
 * 基本的に一切のサポートを行いません
 * 何か問題が起きた時、ご自身で解決できる方のみご利用ください。
 */

(function() {

'use strict';
const script = document.currentScript;
const param = PluginManagerEx.createParameter(script);

Game_Battler.prototype.regenerateHp = function() {
    const minRecover = -this.maxSlipDamage();
    var value = Math.floor(this.mhp * this.hrg);
   
    // 最低値の計算があるので、先に最大HPに対してのスリップダメージを計算
    if (value !== 0) {
        if (value < 0){ 
            const SlipDivision = this.getSlipDivision();
            const SlipPush =  this.getSlipPush();
            
            if(SlipDivision > 0){
                value = Math.round(value / SlipDivision);
            } 
            if(SlipPush > 0){
                value = Math.round(value * SlipPush);
            }
        }
    }
    
    // 最低値の計算
    value = Math.max(value, minRecover);

    // ダメージの適用と、負のダメージの場合はアニメ再生
    if (value !== 0) {
        if (value < 0){ 
            var animationId = param.SlipDamAnime;
            const animation = $dataAnimations[animationId];
            if (animation) {
                $gameTemp.requestAnimation([this], animationId);
            }
        }
        this.gainHp(value);
    }
}

/* アクターのスリップ軽減率取得 */
Game_Actor.prototype.getSlipDivision = function() {
    
    var lastMagni = 0;
    var getMeta = 0;
    
    var states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.SlipDivision){
            getMeta = states[i].meta.SlipDivision;
            if(getMeta > 0) lastMagni += Number(getMeta);
        }
    }
    if (this.actor() && this.actor().meta && this.actor().meta.SlipDivision) {
        getMeta = this.actor().meta.SlipDivision;
        if(getMeta > 0) lastMagni += Number(getMeta);
    }
    if (this.currentClass() && this.currentClass().meta && this.currentClass().meta.SlipDivision)  {
        getMeta = this.currentClass().meta.SlipDivision;
        if(getMeta > 0) lastMagni += Number(getMeta);
    }
    
    var equips = this.equips();
    for (let i = 0; i < equips.length; i++) {      
        if (equips[i] && equips[i].meta && equips[i].meta.SlipDivision){
            getMeta = equips[i].meta.SlipDivision;
            if(getMeta > 0) lastMagni += Number(getMeta);
        }
    }
    
    return lastMagni;
};

/* エネミーのスリップ軽減率取得 */
Game_Enemy.prototype.getSlipDivision = function() {
    
    var lastMagni = 0;
    var getMeta = 0;
    
    var states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.SlipDivision){
            getMeta = states[i].meta.SlipDivision;
            if(getMeta > 0) lastMagni += Number(getMeta);
        }
    }
    if (this.enemy() && this.enemy().meta && this.enemy().meta.SlipDivision) {
        getMeta = this.enemy().meta.SlipDivision;
        if(getMeta > 0) lastMagni += Number(getMeta);
    }
    
    return lastMagni;
};

/* アクターのスリップ倍加率取得 */
Game_Actor.prototype.getSlipPush = function() {
    
    var lastMagni = 0;
    var getMeta = 0;
    
    var states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.SlipPush){
            getMeta = states[i].meta.SlipPush;
            if(getMeta > 0) lastMagni += Number(getMeta);
        }
    }
    if (this.actor() && this.actor().meta && this.actor().meta.SlipPush) {
        getMeta = this.actor().meta.SlipPush;
        if(getMeta > 0) lastMagni += Number(getMeta);
    }
    if (this.currentClass() && this.currentClass().meta && this.currentClass().meta.SlipPush)  {
        getMeta = this.currentClass().meta.SlipPush;
        if(getMeta > 0) lastMagni += Number(getMeta);
    }
    if (typeof this.equips() !== 'undefined') {
        var equips = this.equips();
        for (let i = 0; i < equips.length; i++) {      
            if (equips[i] && equips[i].meta && equips[i].meta.SlipPush){
                getMeta = equips[i].meta.SlipPush;
                if(getMeta > 0) lastMagni += Number(getMeta);
            }
        }
    }
    return lastMagni;
};

/* エネミーのスリップ倍加率取得 */
Game_Enemy.prototype.getSlipPush = function() {
    
    var lastMagni = 0;
    var getMeta = 0;
    
    var states = this.states();
    for (let i = 0; i < states.length; i++) {
        if (states[i] && states[i].meta && states[i].meta.SlipPush){
            getMeta = states[i].meta.SlipPush;
            if(getMeta > 0) lastMagni += Number(getMeta);
        }
    }
    if (this.enemy() && this.enemy().meta && this.enemy().meta.SlipPush) {
        getMeta = this.enemy().meta.SlipPush;
        if(getMeta > 0) lastMagni += Number(getMeta);
    }
    
    return lastMagni;
};


})();
